/*
 * $ soa.h Release 12.2.1.3.0 $
 */

/* Copyright (c) 1989, 2017, Oracle and/or its affiliates. 
All rights reserved.*/

/*
NAME
   SOA.H - System-dependent routines and macros for generic OA
CONSTANTS
   None.
MACROS
   soamalc - System-dependent ALloCate memory
   soafree - System-dependent FREe memory
   soareal - System-dependent ReALlocate memory
   soacalc - System-dependent CALloC memory
   soakill - System-dependent Kill process
TYPEDEFS
   None.
STRUCTURES
   None.
PROTOTYPES
   soasigc - System-dependent catch signal (callback)
   soasigj - System-dependent catch signal (longjmp)
   soasigb - System-dependent signal block/unblock
   soagenv - System-dependent Get string from the ENVironment
   soaghom - System-dependent Get users HOMe
   soaexec - System-dependent execute command
   soaexcv - System-dependent spawn Oracle process
   soawait - System-dependent wait for child process
   soafifl - System-dependent find file
   soanyau - System-dependent notify OS user
   soafork - System-dependent fork a child process

   soatafn - System-dependent translate file name under ORACLE_HOME
   soadn   - System-dependent display msg. #
   soads   - System-dependent display msg. string
   soapath - System-dependent path construct
   soapthgn - System-dependent path generate (a bit more generic)

   soafl.c - FiLe functions
   soaflin  - System-dependent file initialize and open
   soafllr  - System-dependent file line read
   soaflre  - System-dependent file read
   soaflwr  - System-dependent file write
   soaflex  - System-dependent file exit and close
   soaflgp  - System-dependent file get position
   soaflsp  - System dependent file set position
   soaflvx  - System-dependent file Verify eXistence
   soaflvd  - System-dependent file Verify Directory
   soaflgs  - System-dependent file Get Size
   soaflrm  - System-dependent file ReMove (unlink)
   soaflrn  - System-dependent file ReName
   soaflfl  - System-dependent file FLush
   soafl2f  - converts FILE * to soafl struct for use with other
              soafl functions (if possible)

   soafd.c - File Directory functions
   soafdgc  - System-dependent directory Get Current
   soafdo   - System-dependent directory Open
   soafdn   - System-dependent Next file name in directory 
   soafdc   - System-dependent directory Close
   soafdnew - System-dependent directory NEW
   soafdrm  - System-dependent Directory ReMove

   soa2ts.c - TWO_TASK related functions
   soa2tsc  - System-dependent 2 TaSk Check
   soa2tsg  - System-dependent 2 TaSk Get

NOTES
   None.
OWNER
DATE
MODIFIED
*/

#ifndef SOA
#define SOA

#ifndef S
#include <s.h>
#endif /* S */

#ifndef SL
#include <sl.h>
#endif /* SL */

#ifndef OA
#include <oa.h>
#endif /* OA */

#ifndef LDX
#include <ldx.h>
#endif

#include <signal.h>

#ifdef DESKTOP
/*----------------- HACK FOR SYSTEMS WITHOUT STDOUT AND STDERR --------------*/
#include <stdio.h>

/* Was #ifdef NEVER             */
#if defined(NEVER) && !defined(WIN_NT)
externref FILE *stdout;
externref FILE *stderr;
#endif

#endif


/* Was #ifdef MSWINDOWS         */
#if defined(MSWINDOWS) && !defined(WIN_NT)
#define HUGE     _huge
#endif


#ifdef MAC

#ifndef SUTNCMALLOC
#include <sutncmalloc.h>
#endif

#endif

/*
** Structures
*/

/*
** System-dependent directory conteXt
*/
#define soarx soafdx    /* until everyone adopts name change */
typedef dvoid *soafdx;

/* System dependent file operations control structure.  This structure
** contains the flags and fields necessary to perform system dependent file
** functions like open, read, write, and close.
*/
struct soafl
{
  sword soaflsta;   /* Current status of file */
  ub4   soaflflg;   /* Flags for file operations */
  text  *soaflnam;  /* Fully qualified file name that you are working with */
  text  *soaflftx;  /* Text flags for operation */
  dvoid *soaflfpt;  /* Current file pointer */
};

typedef struct soafl soafl;

/*
** Constants
*/
/* For soadn and soads */
#define SOADOUT 1	/* Stdout */
#define SOADERR 2	/* Stderr */

/* For soadn and soads */
#define SOADCHAR 1	/* Printed only as text - (%c) */
#define SOADINT  2	/* Printed only as eb4  - (%ld) */
#define SOADTEXT 3	/* Printed only as text * - (%s) */

/* Flags for soaflin (file init) to tell the function how to open the file */
#define SOAFL_READ          (ub4)0x0001
#define SOAFL_WRITE         (ub4)0x0002
#define SOAFL_BINARY        (ub4)0x0004
#define SOAFL_APPEND        (ub4)0x0008
#define SOAFL_RDWR          (ub4)0x0010

/*
** Flags for soaflgs that determine which forks to use in calculating
** the size of a file.
*/
#define SOAFL_SIZ_DAT       (ub2)0x01
#define SOAFL_SIZ_RSRC      (ub2)0x02

/* System dependent strings to open the file correctly on each platform */
#define SOAFL_READ_STR      "r"     /* open a file for reading */
#define SOAFL_WRITE_STR     "w"     /* open a file for writing */
#define SOAFL_RDWR_STR      "r+"    /* open a file for reading and writing */
#define SOAFL_BINARY_STR    "b"     /* open a file of binary type */
#define SOAFL_APPEND_STR    "a"     /* open a file to be appended to */

/* Error codes for file handling functions */
#define SOAFL_BAD_FN	    -1     /* File name was invalid */
#define SOAFL_EOF	    0      /* End of file reached */

/* process ids should be declared as: text pid[SOA_PID_SZ] */
#define SOA_PID_SZ	    12

/* signal parameter to soakill, soasig* */
#define SOA_SIG_TEST	    0		/* to check if pid is running */
#define SOA_SIG_KILL	    SIGKILL	/* forced kill */
#define SOA_SIG_TERM	    SIGTERM	/* terminate */
#define SOA_SIG_INT	    SIGINT	/* interrupt */

#if defined(WIN_NT)     
# define SIGUSR1            30      /* user defined*/
# define SIGUSR2            31      /* user defined*/
/*# define slsig_t            VOID    Signal return type*/

/* Event name*/
# define OFC_SIGUSR1 ":SIGUsr1"
# define OFC_SIGUSR2 ":SIGUsr2"
# define OFC_SIGTERM ":SIGTerm"
# define OFC_SIGINT  ":SIGInt"
#endif  /*#if defined(WIN_NT)*/

#ifndef MSWINDOWS
# define SOA_SIG_USER1	    SIGUSR1	/* user defined 1 */
# define SOA_SIG_USER2	    SIGUSR2	/* user defined 2 */
#else
# define SOA_SIG_USER1	    30		/* user defined 1 */
# define SOA_SIG_USER2	    31		/* user defined 2 */
#endif

/* Flags for soapthgn */
#define SOA_INSEXT	    0x1
#define SOA_HOST	    0x2
#define SOA_ABSDIR	    0x4

/* name of the home directory */
#ifndef DESKTOP
#define SOA_HOME            (CONST text *)"HOME"
#else
#define SOA_HOME            (CONST text *)"OAHOME"
#endif

/*
**  Location and names of files used for line printing code (oanp)
*/
#ifdef V1_BACKEND
# define SOANPPLOC "?/mail/printers"         /* printer definition dir */
# define SOANPSLOC "?/mail/admin"            /* spool command dir      */
#else
# define SOANPPLOC "?/oacore21/admin/printers" /* printer definition dir */
# define SOANPSLOC "?/oacore21/admin/resource" /* spool command dir      */
#endif
#define SOANPSCMD "spoolcmd.sh"              /* spool command name     */

#ifndef UNIX
# ifndef DESKTOP

   /* These should be defined for other ports */
#  undef SOANPPLOC
#  undef SOANPSLOC
#  undef SOANPSCMD

# endif
#endif

/*
** Macros
*/

/*
**  SET SLERC ERROR INFO -
**
**  Put specified OA error into slerc and put global "errno" in sle if
**  this platform supports it.
*/
#ifdef UNIX
# define SOA_SET_SLERC(sle, res) { CLRSTRUCT(sle); \
                                    SLERC_OERC(sle) = (eword)(res); \
                                    (sle).slercerrno = errno; }
#else
# ifdef DESKTOP
#  define SOA_SET_SLERC(sle, res) { CLRSTRUCT(sle); \
                                     SLERC_OERC(sle) = (eword)(res); \
                                     (sle).slercose = errno; }
# else
#  define SOA_SET_SLERC(sle, res) { CLRSTRUCT(sle); \
                                     SLERC_OERC(sle) = (eword)(res); }
# endif
#endif


/*
** soamalc - System-dependent ALloCate memory call
** soahmlc - System-dependent ALloCate memory call - handle base, sb4 size
** soa2mlc - System-dependent ALloCate memory function name
**	     passed as a parameter to other functions
**
** Parameters (in):
**
**   size - number of bytes desired
**
** Returns:
**
**   dvoid * - pointer to the allocated area of memory
**
** Notes:
**
**   None.
*/
#ifdef MSWINDOWS
# define SOAHM_DEF_HNDL(hndl) ub4 (hndl);     /* handle variable definition */
# define SOAHM_CLR_HNDL(hndl) ((hndl) = 0)

  dvoid * _loadds soamalc(size_t size);
  dvoid HUGE *soahma(ub4 *phndl, sb4 size, eword zero);
# define soahmlc(h,s)	soahma(&((ub4)h), (sb4)(s), FALSE)
# define soa2mlc	soamalc
#else
# define SOAHM_DEF_HNDL(hndl)                 /* don't append semi-colon    */
# define SOAHM_CLR_HNDL(hndl)
# ifdef MAC
#  define soamalc	sutNCMalloc
#  define soahmlc(h,s)	sutNCMalloc((size_t)s)
#  define soa2mlc	sutNCMalloc
# else
#  define soamalc	malloc
#  define soahmlc(h,s)	malloc((size_t)s)
#  define soa2mlc	malloc
# endif
#endif


/*
** soacalc - System-dependent Clear and ALloCate memory call
** soahclc - System-dependent Clear and ALloCate memory call - handle base
** soa2clc - System-dependent Clear and ALloCate memory function name
**	     passed as a parameter to other functions
**
** Parameters (in):
**
**   nelem  - number of elements
**   elsize - size of each element
**
** Returns:
**
**   dvoid * - pointer to the allocated area of memory
**
** Notes:
**
**   soacalc zeros out the buffer before returning it.
*/
#ifdef MSWINDOWS
# define soacalc(nelem,elsize)   calloc((nelem), (elsize))
# define soahclc(h,n,s)          soahma(&((ub4)h), (sb4)(s*n), TRUE)
# define soa2clc                 calloc
#else
# ifdef MAC
#  define soacalc(nelem,elsize) \
            sutNCCalloc((size_t)(nelem), (size_t)(elsize))
#  define soahclc(h,n,s)	\
            sutNCCalloc((size_t)(n), (size_t)(s))
#  define soa2clc		 sutNCCalloc
# else
#  define soacalc(nelem,elsize)   calloc((nelem), (elsize))
#  define soahclc(h,n,s)	 calloc((n), (s))
#  define soa2clc		 calloc
# endif
#endif


/*
** soafree - System-dependent FREe memory call
** soahfre - System-dependent FREe memory call - handle base, sb4 size
** soa2fre - System-dependent FREe memory function name
**	     passed as a parameter to other functions
**
** Parameters (in):
**
**   pointer - pointer to the memory to free
**
** Returns:
**
**   Nothing.
**
** Notes:
**
**   None.
*/
#ifdef MSWINDOWS
# define soafree	free
  void soahmf(ub4 hndl);
# define soahfre(h,p)	soahmf((ub4)(h))
# define soa2fre	free
#else
# ifdef MAC
#  define soafree(ptr)	sutNCFree((dvoid *)(ptr))
#  define soahfre(h,p)	sutNCFree((dvoid *)(p))
#  define soa2fre	sutNCFree
# else
#  define soafree	free
#  define soahfre(h,p)	free(p)
#  define soa2fre	free
# endif
#endif

/*
** soareal - System-dependent ReALlocate memory call
** soahrea - System-dependent ReALlocate memory call - handle base, sb4 size
** soa2rea - System-dependent ReALlocate memory function name
**	     passed as a parameter to other functions
**
** Parameters (in):
**
**   pointer - pointer to the memory to reallocate
**   size    - number of bytes desired
**
** Returns:
**
**   dvoid * - pointer to the allocated area of memory
**
** Notes:
**
**   None.
*/
#ifdef MSWINDOWS
# define soareal(ptr,size)	realloc((ptr),(size))
  dvoid HUGE *soahmr(ub4 *phndl, sb4 size);
# define soahrea(h,p,size)	soahmr(&((ub4)h), (sb4)(size))
# define soa2rea		realloc
#else
# ifdef MAC
#  define soareal(ptr,size) \
             sutNCRealloc((dvoid *)(ptr),(size_t)(size))
#  define soahrea(h,ptr,size) \
             sutNCRealloc((dvoid *)(ptr),(size_t)(size))
#  define soa2rea		sutNCRealloc
# else
#  define soareal(ptr,size)	realloc((ptr),(size))
#  define soahrea(h,p,size)	realloc((p),(size))
#  define soa2rea		realloc
# endif
#endif

/*
**  Directory Separator Character
*/
#ifdef MSWINDOWS
#   define SOAFL_SEP_STR '\\'
#else
# ifdef MAC
#   define SOAFL_SEP_STR ':'
# else
#   define SOAFL_SEP_STR '/'
# endif
#endif


/*
** soahmcp - System-dependent memory copy for HUGE buffers
**
** Parameters (in):
**
**   dst  - destination
**   src  - source
**   size - number of bytes to copy
**
** Returns:
**
**   None.
**
** Notes:
**
**   None.
*/
#ifdef MSWINDOWS
  dvoid HUGE *soahmcp(dvoid HUGE *dst, dvoid HUGE *src, sb4 size);
#else
# define soahmcp(d,s,c) (dvoid HUGE *)memcpy((dvoid *)d,(dvoid *)s,(size_t)c)
#endif

/*
** soahmwr - System-dependent file WRite for HUGE buffers
**
** Parameters:
**   sle     - (OUT) system dependent error struct (UNUSED)
**   fp      - (IN)  opaque file handle (currently just FILE *)
**   buf     - (IN)  huge buffer
**   bufsize - (IN)  number of bytes to write out
**   bytes   - (OUT) actual number of bytes written
**
** Returns:
**   OA_SUCC or OA_FAIL
**
** Notes:
**   This routine doesn't use hwrite() since that function requires an HFILE
** as the file pointer.
*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
sword soahmwr( /*_ slerc *sle, FILE *fp, CONST dvoid HUGE *buf,
                   sb4 bufsize, sb4 *bytes _*/ ); 
#endif

/*
** soahmrd - System-dependent file ReaD for HUGE buffers
**
** Parameters:
**   sle     - (OUT) system dependent error struct (UNUSED)
**   fp      - (IN)  opaque file handle (currently just FILE *)
**   buf     - (IN)  huge buffer
**   bufsize - (IN)  number of bytes to read in
**   bytes   - (OUT) actual number of bytes read
**
** Returns:
**   OA_SUCC or OA_FAIL
**
** Notes:
**   This routine doesn't use hread() since that function requires an HFILE
** as the file pointer.
*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
sword soahmrd(/*_ slerc *sle, FILE *fp, CONST dvoid HUGE *buf,
                   sb4 bufsize, sb4 *bytes _*/);
#endif


/*
** soakill - System-dependent Kill process
**
** Parameters (in):
**
**   pid   - OS process id
**   sig   - signal number to kill process with (defined above)
**
** Returns:
**
**   sword - 0 for success,  -1 for failure
**
** Notes:
**
**   None.
*/
#ifdef lint
sword soakill(/*_ CONST text *pid, eword sig _*/);
#else
#if !defined(WIN_NT)             
# define soakill(pid, sig) \
        (sword)kill(atoi((char*)pid), sig)
# else
sword NTSetEvent(/* DWORD dwPID, eword sig */);
# define soakill(pid, sig) \
        (sword)NTSetEvent((DWORD)atoi((char*)pid), sig)
# endif     /* #if !defined(WIN_NT)*/
#endif


/*
** PROTOTYPES
*/

/*
** soasigc - System-dependent catch signal (callback)
**
** Parameters (in):
**
**   sig     - signal number to catch (defined above)
**   callp   - callback procedure
**   callc   - callback procedure context
**
** Notes:
**
**   A handler is installed to catch signal sig and to invoke (*callp)().
**   Further occurences of sig are blocked during the callback.
*/
#if !defined(WIN_NT)        
void soasigc( /*_ eword sig, void (*callp)(ptr_t callc, eword sig),
		  ptr_t callc _*/ );
#else
sword soasigc( /*_ eword sig, void (*callp)(ptr_t callc, eword sig),
		  ptr_t callc _*/ );

#endif  /*#if !defined(WIN_NT)*/

/*
** soasigj - System-dependent catch signal (longjmp)
**
** Parameters (in):
**
**   sig     - signal number to catch (defined above)
**   jmpbuf  - long jump buffer
**
** Notes:
**
**   A handler is installed to catch signal sig and do an slgjmp(jmpbuf, sig).
**   Signal sig remains blocked until an explicit soasigb(sig, FALSE) by user.
*/
#if !defined(WIN_NT)        
void soasigj( /*_ eword sig, slgbuf *jmpbuf _*/ );
#else
sword soasigj( /*_ eword sig, slgbuf *jmpbuf _*/ );
#endif  /*#if !defined(WIN_NT)*/

/*
** soasigb - System-dependent signal block/unblock
**
** Parameters (in):
**
**   sig     - signal number (defined above)
**   block   - block (TRUE) or unblock (FALSE)
**
** Returns:
**
**   boolean - previously blocked (TRUE) or unblocked (FALSE)
**
** Notes:
**
**   Can't block SOA_SIG_KILL.
*/
boolean soasigb( /*_ eword sig, boolean block _*/ );

/*
** soagenv - System-dependent Get string from the ENVironment
**
** Parameters (in):
**
**   env     - environment variable to look for
**   buf     - buffer to return the value in
**   bufsiz  - size of the buffer
**
** Returns:
**
**   sword   - returns 0 is found and -1 if not found.
**
** Notes:
**
**   None.
*/
sword soagenv( /*_ CONST text *env, text *buf, size_t bufsiz _*/ );

/*
** soaghom - System-dependent Get users HOMe
**
** Parameters (in):
**
**   buf     - buffer to return the value in
**   bufsiz  - size of the buffer
**
** Returns:
**
**   sword   - returns 0 is found and -1 if not found.
**
** Notes:
**
**   None.
*/
sword soaghom( /*_ text *buf, size_t bufsiz _*/ );

/*
** soaexec - System-dependent EXECute command
**
** Parameters (in):
**
**   cmd     - command line to run
**   dowait  - wait for command to finish?
**
** Returns:
**
**   se      - serc returns error info
**
** Notes:
**
**   None.
*/
void soaexec( /*_ serc *se, CONST char *cmd, int dowait _*/ );

/*
** soaexcv - System-dependent EXECute command
**
** Parameters (in):
**
**   filename- name of object file to execute
**   pargv   - arguments to process
**   detach  - run command as detached process
**
** Returns:
**
**   OA_FAIL - failed to start process (error info in se)
**   OA_SUCC - process started (child pid returned in pid)
**
** Notes:
**
**   None.
*/
/*  Since Createprocess() only take command string instead of */
/* array of string.*/
#if !defined(WIN_NT)
sword soaexcv(/*_ serc *se, CONST text *filename, text *pargv[],
                  boolean detach, text *pid _*/);
#else
sword soaexcv(/*_ serc *se, CONST text *filename, text *pargv,
                  boolean detach, text *pid _*/);
#endif  /* if !defined(WIN_NT)*/

/*
** soawait - System-dependent wait for child process
**
** Returns:
**
**   OA_FTL  - wait operation failed
**   OA_FAIL - child was not successfuly spawned
**   OA_WARN - child exited with nonzero error code
**   OA_SUCC - child exited with no error
**   pid     - exited child pid, if any, is returned
**
** Notes:
**
**   None.
*/
sword soawait(/*_ serc *se, text *pid _*/);

/*
** soafifl - Cover routine for slfindfile
**
** Description:
**
**   This routine initializes the context for slfindfile, if neccessary and
**   calls slfindfile to locate a file which matches the passed in
**   description.  NOTE: This routine does not open the file.
**
** Parameters:
**   oaxm;      - Global context manager
**   prod;      - Product name
**   version;   - Version of the product
**   name;      - Base name of the file
**   type;      - type of the file
**   langdep;   - Language dependent
**   guidep;    - GUI dependent
**   buf;       - Output buffer
**   buflen;    - Size of the buffer
**
** Returns:
**   OA_SUCC	- if successfully finds the file
**   OA_FAIL  	- if it don't
*/
sword soafifl(/*_ struct oaxmctx *oaxm, CONST text *prod, CONST text *ver,
                  CONST text *name, eword type, boolean langdep,
                  boolean guidep, text *buf, size_t buflen _*/);

/*
**
** soanyau - Notify an OS User
**
** Returns:
**
**   OA_FAIL - unable to notify the user
**       cnt - number of successful messages sent
**
** Notes:
**    
**   None.
*/
sword soanyau(/*_ text *user, text *hostname, text *outbuf _*/);

/*
**
** soafork - fork a child process
**
** Returns:
**
**   0 - to the child process
**   process id of the child to the parent process
**
** Notes:
**    
**   None.
*/
#ifdef DESKTOP
sword soafork();
#else
#define soafork() (sword)fork()
#endif

/*
** soaflin - System-dependent file initialize
**
** Returns:
**
**   OA_FAIL - file was not successfully opened
**   OA_WARN - you are opening a file for writing that already exists
**   OA_SUCC - file was successfully opened
**
** Notes:
**
**   None.
*/
sword soaflin(/*_ soafl **file_hdl, CONST text *file_name, ub4 open_flags _*/);

/*
** soafllr - System-dependent file line read
**
** Returns:
**
**   OA_FAIL - file was not successfully read
**   OA_SUCC - file was successfully read
**
** Notes:
**
**   None.
*/
sword soafllr(/*_ soafl *file_hdl, dvoid *read_buffer, ub4 buffer_len _*/);

/*
** soaflre - System-dependent file read
**
** Returns:
**
**   OA_FAIL - file was not successfully read
**   OA_WARN - 
**   OA_SUCC - file was successfully read
**
** Notes:
**
**   None.
*/
sword soaflre(/*_ soafl *file_hdl, dvoid *read_buffer, ub4 buffer_len _*/);

/*
** soaflwr - System-dependent file write
**
** Returns:
**
**   OA_FAIL - file was not successfully written
**   OA_WARN - 
**   OA_SUCC - file was successfully written
**
** Notes:
**
**   None.
*/
sword soaflwr(/*_ soafl *file_hdl, dvoid *write_buffer, ub4 buffer_len _*/);

/*
** soaflex - System-dependent file close
**
** Returns:
**
**   OA_FAIL - file was not successfully closed
**   OA_WARN - 
**   OA_SUCC - file was successfully closed
**
** Notes:
**
**   None.
*/
sword soaflex(/*_ soafl *file_hdl _*/);

/*
** soaflgp - System-dependent file get position
**
** Returns:
**
**   OA_FAIL - 
**   OA_WARN - 
**   OA_SUCC - position was successfully obtained
**
** Notes:
**
**   None.
*/
sword soaflgp( /*_ soafl *file_hdl, sb4 *pos _*/ );

/*
** soaflgp - System-dependent file set position
**
**   Set file stream to offset from beginning of file
**
** Returns:
**
**   OA_FAIL - position could not be set.
**   OA_WARN - 
**   OA_SUCC - stream was successfully set to pos.
**
** Notes:
**
**   None.
*/
sword soaflsp( /*_ soafl *file_hdl, sb4 pos _*/ );

/*
**  SOAFLVX - Verify eXistence
**    Verifies that a particular file exists.  Can be used to validate a file
**  name field or for bringing up a dialog asking if they really want to
**  clobber a file.
*/
sword soaflvx( /*_ slerc *sle, CONST text *file_name _*/ );

/*
**  SOAFLVD - Verify Directory
**    Verifies that a particular file exists and that it is a directory.
*/
sword soaflvd( /*_ slerc *sle, CONST text *dir_name _*/ );

/*
**  SOAFLGS - Get Size
**    Gets the size of a file in bytes.  Can also act as a validation
**  function.
*/
sword soaflgs( /*_ slerc *sle, CONST text *file_name, ub4 *sizeP, ub2 sfl _*/ );

/*
**  SOAFLRM - ReMove (or unlink) a file
**    Removes the specified file.
*/
sword soaflrm( /*_ slerc *sle, CONST text *file_name _*/ );

/*
**  SOAFLRN - ReName a file
**    Renames the specified file.
*/
sword soaflrn( /*_ slerc *sle, CONST text *oldname, CONST text *newname _*/ );

/*
**  SOAFLTN -  OSD File Temporary Name
**    This function generates the name of a temporary file.
*/
text *soafltn( /*_ slerc *sle, CONST text *dir, CONST text *fprefix,
                   CONST text *fsuffix, text *buf, size_t bufsize _*/ );

/*
** Soaflfl - System-dependent FiLe FLush
*/
sword soaflfl( /*_ slerc *sle, soafl *file_hdl _*/ );

/*
** Soafl2f - Converts FILE to soafl struct for use with other soafl funcs
*/
void soafl2f( /*_ FILE *file_ptr, soafl *file_hdl _*/ );

/*
**  SOAFLCFP - Construct Full Path
*/
sword soaflcfp( /*_ text *full, ub2 len, CONST text *base _*/ );

/*
** soatafn - System-dependent TrAnslate File Name under ORACLE_HOME 
**
** Returns:
**
**   OA_FAIL - file was not successfully translated
**   OA_SUCC - file was successfully translated
**
** Notes:
**
**   None.
*/
sword soatafn(/*_ text *filename, text *out, size_t outlen, text *prod,
		  text *subdir, size_t *reslen _*/);

/*
** soadn - System-dependent Display msg. # 
**
** Returns:
**
**   OA_FAIL - if not successfully displayed msg. 
**   OA_SUCC - if successfully displayed msg. 
**
** Notes:
**
**   Gets msg. string from lms and then calls soads. 
*/
sword soadn ( /*_ struct oamh *mhx, eword msgno, eword prodid, eword where _*/);

/*
** NAME: soads() System-dependent Display msg. String
**
**    Display an error msg. string in a system dependent fashion
**
** FUNCTION:
**
**    Args. are passed in as, 
**
**    soads (SOADOUT, 2, <type><arg><type><arg>)
**
**    where type is an int and is of one of the macros above. Arg. is of the
**    corresponding type.
**
** RETURNS: Success or failure.
*/
sword soads VAFP (( eword where,  eword numargs, ...));

/*
** soafdgc - System-dependent directory Get Current
**
** Returns:
**      current directory name
**
** Notes:
**
*/
sword soafdgc( /*_ text *dirname _*/ );

/*
** soafdo - System-dependent directory Open 
**
** Returns:
**	directory conteXt if successful or null conteXt
**
** Notes:
**
*/
#define soaro soafdo    /* until everyone adopts name change */
soafdx soafdo ( /*_ CONST text *dirname _*/); 

/*
** soafdn - System-dependent get Next filename in directory 
**
** Returns:
** 	next file name or null if at end of directory.
**
** Notes:
**
*/
#define soarn soafdn    /* until everyone adopts name change */
CONST text *soafdn ( /*_ soafdx dirx _*/); 

/*
** soafdc - System-dependent directory Close 
**
** Returns:
**	OA_SUCC, or OA_FAIL	
**
** Notes:
**
*/
#define soarc soafdc    /* until everyone adopts name change */
sword soafdc ( /*_ soafdx dirx _*/); 


/*
** soafdnew - System-dependent directory NEW
**
** Returns:
**	OA_SUCC, or OA_FAIL	
**
** Notes:
**
*/
#define soarnew soafdnew    /* until everyone adopts name change */
sword soafdnew ( /*_ CONST text *dirname _*/); 

/*
** soafdrm - System-dependent Directory ReMove
**
** Parameters:
**   sle      - (OUT) system dependent error struct (UNUSED)
**   dirname  - (IN)  name of directory to remove
**   contents - (IN)  if TRUE delete contents of directory as well
** Returns:
**      OA_SUCC, or OA_FAIL
** Notes:
*/
sword soafdrm ( /*_ slerc *sle, CONST text *dirname, boolean contents _*/); 

/*
** soapath - System-dependent path construct
**
** Description:
**   Builds a system dependent pathname under ORACLE_HOME.
**   Under UNIX, it will be of the form:
**	$ORACLE_HOME/product/area/filename[.extension]
**
** Parameters:
**   path	- pointer to user buffer where path is returned
**   pathsz	- size of path buffer
**   product	- product name (i.e. "mail")
**   area	- area under product (i.e. "log")
**   filename	- filename component (i.e. "omdaemon")
**   extension	- filename extension component (i.e. "log" or "")
**   instanceid - instantiation of object
**   flags	- actions
**	bit 0	= use instanceid before extension
**	bit 1	= hostname prepended w/ '_'  (e.g., impala_<rest>)
**
** Returns:
**   OA_SUCC	- if successfully translated and built pathname
**   otherwise	- sltln error
**
*/
sword soapath(/*_ text *path, size_t pathsz, CONST text *product,
		CONST text *area, CONST text *filename, CONST text *extension,
		ub2 instanceid, ub2 flags _*/);

/*
**  Name: soapthgn - Construct a system dependent path using generic 
**                   directory components.
**
**  Description:
**    Construct a system dependent path ala soapath() without using 
**    product dependent components.
**    
**    Under UNIX, construct the following:      <bdir>/<state>/<fname>.<ext>
**
**  Parameters:
**    path      - constructed path
**    pathsz    - size of buffer
**    bdir      - base directory                REQUIRED
**    state     - state component of directory
**    fname     - file name                     REQUIRED
**    ext       - file extension
**    instance  - instance id for server logs 
**    flags     - control flags
**                SOA_INSEXT  -: use instance before extension
**                SOA_HOST    -: hostname prepended w/ '_'  
**                               (e.g., impala_<rest>)
**  Returns:
**    OA_SUCC   - if path succesfully constructed 
**    OA_FAIL   - if sltln error occurred
*/

sword soapthgn(/*_ text *path, size_t pathsz, text *bdir, text *state,
		   text *fname, text *ext, ub2 instance, ub4 flags _*/);

/*
**  Name: soapfil - Return the name and extension components of a filename
**
**  Description:
**    Take a system dependent file name and separate its component parts
**    
**    Under UNIX, something like "document.txt" returns "document" as the
**    root portion of the file and "txt" as the extension
**
**  Parameters:
**    buf	- Working buffer
**    bufsz	- size of buffer
**    froot	- root of file name
**    ext	- extension of file name
**    fname	- filename to be parsed
**
**  Returns:
**    OA_SUCC	- if file components were successfully separated
**    OA_FAIL	- if parsing error occured
*/
sword soapfil(/*_ text *buf, size_t bufsz, text **froot, text **ext, 
		  text *fname _*/);

/*
** soa2tsc  - System-dependent 2 TaSk Check
**
** Parameters (in):
**
**	None
**
** Returns:
**
**   boolean - TRUE if TWO_TASK is set, FALSE otherwise.
**
** Notes:
**
**   None.
*/
boolean soa2tsc();

/*
** soa2tsg  - System-dependent 2 TaSk Get
**
** Parameters (in):
**
**   buf     - buffer to return the value in
**   bufsiz  - size of the buffer
**
** Returns:
**
**   sword   - returns 0 is found and -1 if not found.
**
** Notes:
**
**   None.
*/
sword soa2tsg( /*_ text *buf, size_t bufsiz _*/ );

/*
**  SOAFLDM - Date Modified
**    Gets the last date a file was modified.  For some OS's we may only be
**  able to get the date of file creation so that will be returned instead.
*/
sword soafldm( /*_ slerc *sle, CONST text *file_name, ldx_t *mdate _*/ );
/*
**  SOAFLCP - CoPy file
**    Copies "src" to "dst".  Assumes both src and dst are full pathnames.
*/
sword soaflcp( /*_ slerc *sle, CONST text *src, CONST text *dst,
                   boolean overwrite, boolean append _*/ );

#endif /* SOA */
